<?php

/*
 * This file is part of the SiteOne Crawler.
 *
 * (c) Ján Regeš <jan.reges@siteone.cz>
 */

declare(strict_types=1);

namespace Crawler\Output;

use Crawler\Analysis\Result\UrlAnalysisResult;
use Crawler\Components\SuperTable;
use Crawler\CoreOptions;
use Crawler\ExtraColumn;
use Crawler\HttpClient\HttpResponse;
use Crawler\Result\Status;
use Crawler\Result\Summary\Summary;
use Crawler\Utils;
use Swoole\Table;

class TextOutput implements Output
{

    private string $version;
    private Status $status;
    private CoreOptions $options;
    private bool $printToOutput = true;
    private int $extraColumnsFromAnalysisWidth = 0;
    private int $extraColumnsWidth = 0;

    private int $terminalWidth;
    private bool $compactMode = false;
    private int $progressBarWidth = 0;

    /**
     * Extra columns from analysis that will be added to the table via showAnalyzedVisitedUrlResultAsColumn() in Analyzer
     * @var ExtraColumn[]
     */
    private array $extraColumnsFromAnalysis = [];

    private string $outputText = '';

    private string $originHost;

    /**
     * @param string $version
     * @param Status $status
     * @param CoreOptions $options
     * @param bool $printToOutput
     */
    public function __construct(string $version, Status $status, CoreOptions $options, bool $printToOutput)
    {
        $this->version = $version;
        $this->status = $status;;
        $this->options = $options;
        $this->printToOutput = $printToOutput;
        $this->originHost = parse_url($this->options->url, PHP_URL_HOST);

        $this->extraColumnsWidth = 0;
        foreach ($this->options->extraColumns as $extraColumn) {
            $this->extraColumnsWidth += $extraColumn->getLength() + 3; // 3 = 2 spaces + 1 pipe
        }

        $this->terminalWidth = Utils::getConsoleWidth();
        $this->compactMode = $this->terminalWidth < 140;
        $this->progressBarWidth = $this->options->hideProgressBar ? 0 : ($this->compactMode ? 8 : 26);
    }

    public function addBanner(): void
    {
        // ascii art banner - generated by https://www.asciiart.eu/image-to-ascii :-)
        $banner = "\n";
        $banner .= ' ####                ####             #####        ' . "\n";
        $banner .= ' ####                ####           #######        ' . "\n";
        $banner .= ' ####      ###       ####         #########        ' . "\n";
        $banner .= ' ####     ######     ####       ###### ####        ' . "\n";
        $banner .= '  ######################       #####   ####        ' . "\n";
        $banner .= '    #######    #######       #####     ####        ' . "\n";
        $banner .= '    #######    #######         #       ####        ' . "\n";
        $banner .= '  ######################               ####        ' . "\n";
        $banner .= ' ####     ######     ####              ####        ' . "\n";
        $banner .= ' ####       ##       ####              ####        ' . "\n";
        $banner .= ' ####                ####       ################## ' . "\n";
        $banner .= ' ####                ####       ################## ' . "\n";
        $banner .= "\n";
        $banner .= str_repeat('=', 50) . "\n";

        $texts = [
            'SiteOne Crawler, v' . $this->version,
            'Author: jan.reges@siteone.cz'
        ];

        foreach ($texts as $text) {
            $banner .= '# ' . str_pad($text, 46) . ' #' . "\n";
        }
        $banner .= str_repeat('=', 50);

        // loading the rocket on the ramp and show banner with fancy polynomial delays (exponential delays were to aggressive)
        $lines = explode("\n", $banner);
        $delays = self::getPolynomialDelays(1.2, count($lines));
        $counter = 0;
        foreach ($lines as $line) {
            $this->addToOutput(Utils::getColorText($line, 'yellow') . "\n");

            // add delay between lines
            $usleepTime = intval($delays[$counter] * 1000 * 1000);
            usleep($usleepTime);
            $counter++;
        }

        // the rocket takes off smoothly :)
        usleep(300 * 1000);
        $this->addToOutput("\n");
        usleep(150 * 1000);
        $this->addToOutput("\n");

        if ($this->compactMode) {
            $this->addToOutput(Utils::getColorText("Detected terminal width {$this->terminalWidth} < 140 chars - compact mode activated.\n\n", 'yellow'));
        }
    }

    private static function getPolynomialDelays(float $totalTime, int $iterations, int $power = 2): array
    {
        $delays = [];
        $totalPolySum = 0;

        for ($i = 1; $i <= $iterations; $i++) {
            $totalPolySum += pow($i, $power);
        }

        for ($i = 1; $i <= $iterations; $i++) {
            $delays[] = (pow($i, $power) / $totalPolySum) * $totalTime;
        }

        return $delays;
    }

    public function addUsedOptions(): void
    {
        // $this->addToOutput("Used options: " . Utils::getColorText(print_r($this->options, true), 'gray') . "\n");
    }

    /**
     * @param ExtraColumn[] $extraColumnsFromAnalysis
     * @return void
     */
    public function setExtraColumnsFromAnalysis(array $extraColumnsFromAnalysis): void
    {
        $this->extraColumnsFromAnalysis = $extraColumnsFromAnalysis;
        $this->extraColumnsFromAnalysisWidth = 0;
        foreach ($this->extraColumnsFromAnalysis as $extraColumn) {
            $this->extraColumnsFromAnalysisWidth += $extraColumn->getLength() + 3; // 3 = 2 spaces + 1 pipe
        }
    }

    public function addTableHeader(): void
    {
        $header = str_pad("URL", $this->getUrlColumnSize()) . " |" . " Status " . "|" . " Type     " . "|" . " Time   " . "|" . " Size   ";
        if (!$this->options->hideProgressBar) {
            $header = str_pad(
                    $this->compactMode ? "Progress" : "Progress report",
                    $this->progressBarWidth
                ) . "| " . $header;
        }

        foreach ($this->extraColumnsFromAnalysis as $extraColumn) {
            $header .= " | " . str_pad($extraColumn->name, max($extraColumn->getLength(), 4));
        }

        foreach ($this->options->extraColumns as $extraColumn) {
            $header .= " | " . str_pad($extraColumn->name, max($extraColumn->getLength(), 4));
        }
        $header .= "\n";
        $this->addToOutput(Utils::getColorText($header, 'gray') . str_repeat("-", strlen($header)) . "\n");
    }

    public function addTableRow(HttpResponse $httpResponse, string $url, int $status, float $elapsedTime, int $size, int $type, array $extraParsedContent, string $progressStatus): void
    {
        $isExternalUrl = !str_contains($url, '://' . $this->originHost);
        $urlForTable = !$this->options->showSchemeAndHost && !$isExternalUrl ? (preg_replace('/^https?:\/\/[^\/]+\//i', '/', $url)) : $url;

        $coloredStatus = Utils::getColoredStatusCode($status);
        $contentType = str_pad(Utils::getContentTypeNameById($type), 8);
        $coloredElapsedTime = Utils::getColoredRequestTime($elapsedTime);
        $coloredSize =
            $size > 1024 * 1024
                ? Utils::getColorText(str_pad(Utils::getFormattedSize($size), 6), 'red')
                : str_pad(Utils::getFormattedSize($size), 6);

        $extraHeadersContent = '';
        $extraNewLine = '';
        $extraNewLinePrefix = str_repeat(' ', 2);

        foreach ($this->extraColumnsFromAnalysis as $extraColumn) {
            $value = '';
            $headerName = $extraColumn->name;
            if (array_key_exists($headerName, $extraParsedContent)) {
                $value = $extraParsedContent[$headerName];
            }

            /* @var $value UrlAnalysisResult */
            if ($value instanceof UrlAnalysisResult) {
                $notColorizedLength = strlen($value->toNotColorizedString());
                $strPadContent = str_repeat(' ', max(0, $extraColumn->getLength() - $notColorizedLength));

                $extraHeadersContent .= (' | ' . trim(strval($value)) . $strPadContent);

                if ($this->options->showInlineCriticals && ($criticals = $value->getCritical())) {
                    $extraNewLine .= "{$extraNewLinePrefix}⛔ " . implode("\n{$extraNewLinePrefix}⛔ ", $criticals) . "\n";
                }
                if ($this->options->showInlineWarnings && ($warnings = $value->getWarning())) {
                    $extraNewLine .= "{$extraNewLinePrefix}⚠️ " . implode("\n{$extraNewLinePrefix}⚠️ ", $warnings) . "\n";
                }
            } else {
                $extraHeadersContent .= (' | ' . str_pad($extraColumn->getTruncatedValue($value), max($extraColumn->getLength(), 4)));
            }
        }
        foreach ($this->options->extraColumns as $extraColumn) {
            $value = '';
            $headerName = $extraColumn->name;
            if (array_key_exists($headerName, $extraParsedContent)) {
                $value = trim(strval($extraParsedContent[$headerName]));
            } elseif ($httpResponse->headers && array_key_exists(strtolower($headerName), $httpResponse->headers)) {
                $value = trim($httpResponse->headers[strtolower($headerName)]);
            }

            $extraHeadersContent .= (' | ' . str_pad($extraColumn->getTruncatedValue($value), max($extraColumn->getLength(), 4)));
        }

        if ($this->options->addRandomQueryParams) {
            $urlForTable .= Utils::getColorText('+%random-query%', 'gray');
        }

        if (!$this->options->doNotTruncateUrl) {
            $urlForTable = Utils::truncateInTwoThirds($urlForTable, $this->getUrlColumnSize());
        }

        // put progress to stderr
        $progressContent = '';
        if (!$this->options->hideProgressBar) {
            list($done, $total) = explode('/', $progressStatus);
            if ($this->compactMode) {
                $progressContent = str_pad($progressStatus, 7) . ' |';
            } else {
                $progressToStdErr = sprintf(
                    "%s | %s",
                    str_pad($progressStatus, 7),
                    Utils::getProgressBar(intval($done), intval($total), 10)
                );
                $progressContent = str_pad($progressToStdErr, 17);
            }
        }

        $output = sprintf(
                '%s %s | %s | %s | %s | %s %s',
                $progressContent,
                str_pad($urlForTable, $this->getUrlColumnSize()),
                $coloredStatus,
                $contentType,
                $coloredElapsedTime,
                $coloredSize,
                $extraHeadersContent
            ) . "\n";

        if ($extraNewLine) {
            $output .= rtrim($extraNewLine) . "\n";
        }

        $this->addToOutput($output);
    }

    public function addSuperTable(SuperTable $table): void
    {
        $this->addToOutput("\n");
        $this->addToOutput($table->getConsoleOutput());
    }

    public function addTotalStats(Table $visited): void
    {
        $stats = $this->status->getBasicStats();

        $this->addToOutput("\n");
        $resultHeader = sprintf(
            "Total execution time %s using %s workers and %s memory limit (max used %s)\n",
            Utils::getColorText(Utils::getFormattedDuration($stats->totalExecutionTime), 'cyan'),
            Utils::getColorText(strval($this->options->workers), 'cyan'),
            Utils::getColorText($this->options->memoryLimit, 'cyan'),
            Utils::getColorText(Utils::getFormattedSize(memory_get_peak_usage(true)), 'cyan')
        );
        $this->addToOutput(str_repeat('=', $this->terminalWidth) . "\n");
        $this->addToOutput($resultHeader);
        $this->addToOutput(
            sprintf("Total of %s visited URLs with a total size of %s and power of %s with download speed %s\n",
                Utils::getColorText(strval($stats->totalUrls), 'cyan'),
                Utils::getColorText($stats->totalSizeFormatted, 'cyan'),
                Utils::getColorText(intval($stats->totalUrls / $stats->totalExecutionTime) . " reqs/s", 'magenta'),
                Utils::getColorText(Utils::getFormattedSize(intval($stats->totalSize / $stats->totalExecutionTime), 0) . "/s", 'magenta'),
            )
        );
        $this->addToOutput(
            sprintf(
                "Response times: AVG %s MIN %s MAX %s TOTAL %s\n",
                Utils::getColorText(Utils::getFormattedDuration($stats->totalRequestsTimesAvg), 'magenta'),
                Utils::getColorText(Utils::getFormattedDuration($stats->totalRequestsTimesMin), 'green'),
                Utils::getColorText(Utils::getFormattedDuration($stats->totalRequestsTimesMax), 'red'),
                Utils::getColorText(Utils::getFormattedDuration($stats->totalRequestsTimes), 'cyan')
            )
        );

        $this->addToOutput(str_repeat('=', $this->terminalWidth) . "\n");
    }

    public function addNotice(string $text): void
    {
        $this->addToOutput(Utils::getColorText($text, 'blue') . "\n");
    }

    public function addError(string $text): void
    {
        $this->addToOutput(Utils::getColorText($text, 'red') . "\n");
    }

    public function addSummary(Summary $summary): void
    {
        $this->addToOutput("\n");
        $this->addToOutput($summary->getAsConsoleText());
    }

    public function end(): void
    {
        $this->addToOutput("\n");
    }

    public function addToOutput(string $output): void
    {
        if ($this->printToOutput) {
            echo $output;
        }

        $this->outputText .= $output;
    }

    public function getOutputText(): string
    {
        return $this->outputText;
    }

    public function getType(): OutputType
    {
        return OutputType::TEXT;
    }

    private function getUrlColumnSize(): int
    {
        static $urlColumnSize = null;

        if ($urlColumnSize === null) {
            if ($this->options->urlColumnSize) {
                $urlColumnSize = $this->options->urlColumnSize;
            } else {
                $progressBarWidth = $this->progressBarWidth;
                $statusTypeTimeSizeWidth = 40;
                $extraColumnsWidth = $this->extraColumnsWidth;
                $extraColumnsFromAnalysisWidth = $this->extraColumnsFromAnalysisWidth;
                $freeReserve = 5; // small reserve for unpredictable situations

                $urlColumnSize = $this->terminalWidth
                    - $progressBarWidth
                    - $statusTypeTimeSizeWidth
                    - $extraColumnsWidth
                    - $extraColumnsFromAnalysisWidth
                    - $freeReserve;

                $urlColumnSize = max(20, $urlColumnSize);
            }
        }

        return $urlColumnSize;
    }
}